package gov.va.med.term.cvx.mojo;

import java.io.File;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Optional;
import java.util.UUID;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import gov.va.med.term.cvx.CVXConstants;
import gov.va.med.term.cvx.data.CVXCodes;
import gov.va.med.term.cvx.data.CVXCodes.CVXInfo;
import gov.va.med.term.cvx.data.CVXCodesHelper;
import gov.va.med.term.cvx.propertyTypes.PT_Annotations;
import gov.va.med.term.cvx.propertyTypes.PT_Descriptions;
import gov.va.med.term.cvx.reader.CVXReader;
import gov.va.oia.terminology.converters.sharedUtils.ComponentReference;
import gov.va.oia.terminology.converters.sharedUtils.ConsoleUtil;
import gov.va.oia.terminology.converters.sharedUtils.ConverterBaseMojo;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility;
import gov.va.oia.terminology.converters.sharedUtils.IBDFCreationUtility.DescriptionType;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.BPT_Refsets;
import gov.va.oia.terminology.converters.sharedUtils.propertyTypes.PropertyType;
import gov.va.oia.terminology.converters.sharedUtils.stats.ConverterUUID;
import gov.vha.isaac.MetaData;
import gov.vha.isaac.ochre.api.State;
import gov.vha.isaac.ochre.api.component.concept.ConceptChronology;
import gov.vha.isaac.ochre.api.component.concept.ConceptVersion;
import gov.vha.isaac.ochre.api.constants.DynamicSememeConstants;
import gov.vha.isaac.ochre.model.sememe.dataTypes.DynamicSememeStringImpl;

/**
 * 
 * {@link CVXImportMojo}
 *
 * Goal which converts CVX data into the workbench jbin format
 * 
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
@Mojo (name = "convert-CVX-to-ibdf", defaultPhase = LifecyclePhase.PROCESS_SOURCES)
public class CVXImportMojo extends ConverterBaseMojo
{
	private IBDFCreationUtility importUtil_;

	private HashMap<UUID, String> loadedConcepts = new HashMap<>();

	private PropertyType attributes_;
	private BPT_Refsets refsets_;
	private PT_Descriptions descriptions_;

	private UUID allCvxConceptsRefset;

	private int conceptCount = 0;

	@Override
	public void execute() throws MojoExecutionException
	{
		try
		{
			super.execute();

			//There is no global release date for mvx - but each item has its own date.  This date will only be used for metadata.
			Date date = new Date(); 
			
			importUtil_ = new IBDFCreationUtility(
					Optional.of(CVXConstants.TERMINOLOGY_NAME + " " + converterSourceArtifactVersion),
					Optional.of(MetaData.CVX_MODULES),
					outputDirectory,
					converterOutputArtifactId,
					converterOutputArtifactVersion,
					converterOutputArtifactClassifier, false, date.getTime());
			
			attributes_ = new PT_Annotations();
			descriptions_ = new PT_Descriptions();
			
			refsets_ = new BPT_Refsets(CVXConstants.TERMINOLOGY_NAME);
			refsets_.addProperty("All CVX Concepts");

			// Every time concept created add membership to "All CVX Concepts"
			allCvxConceptsRefset = refsets_.getProperty("All CVX Concepts").getUUID();
			
			final CVXReader importer = new CVXReader(inputFileLocation);
			final CVXCodes terminology = importer.process();
			
			ConsoleUtil.println("Loaded Terminology containing " + terminology.getCVXInfo().size() + " entries");

			/*
			 * Methods from CVXCodes.CVXInfo:
			 * getCVXCode() // float numeric id (CODE?)
			 * getShortDescription() // Required String FSN description?
			 * getFullVaccinename() // Required String preferred term description?
			 * getNotes() // Optional String comment
			 * getOchreState() // Required State (ACTIVE or INACTIVE) 
			 * getLastUpdatedDate(), // Required date ?
			 */
			// Parent cvxMetadata ComponentReference
			final ComponentReference cvxMetadata = ComponentReference.fromConcept(
					createType(MetaData.SOLOR_CONTENT_METADATA.getPrimordialUuid(), "CVX Metadata" + IBDFCreationUtility.metadataSemanticTag_));
			
			// loadTerminologyMetadataAttributes onto cvxMetadata
			importUtil_.loadTerminologyMetadataAttributes(converterSourceArtifactVersion, 
					Optional.empty(), converterOutputArtifactVersion, Optional.ofNullable(converterOutputArtifactClassifier), converterVersion);

			// load metadata
			importUtil_.loadMetaDataItems(Arrays.asList(attributes_, refsets_, descriptions_), cvxMetadata.getPrimordialUuid());
			
			ConsoleUtil.println("Metadata summary:");
			for (String s : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println("  " + s);
			}
			importUtil_.clearLoadStats();

			// Create CVX root concept under ISAAC_ROOT
			final ConceptChronology<? extends ConceptVersion<?>> cvxRootConcept = importUtil_.createConcept(CVXConstants.TERMINOLOGY_NAME, true, MetaData.ISAAC_ROOT.getPrimordialUuid());
			ConsoleUtil.println("Created CVX root concept " + cvxRootConcept.getPrimordialUuid() + " under ISAAC_ROOT");
			
			final UUID fsnSourceDescriptionTypeUUID = PT_Descriptions.Descriptions.ShortDescription.getProperty().getUUID();
			final UUID preferredSynonymSourceDescriptionTypeUUID = PT_Descriptions.Descriptions.FullVaccinename.getProperty().getUUID();

			final UUID cvxCodePropertyUuid = attributes_.getProperty(PT_Annotations.Attribute.CVXCode.getKey()).getUUID();
			final UUID cvxStatusPropertyUuid = attributes_.getProperty(PT_Annotations.Attribute.VaccineStatus.getKey()).getUUID();
			
			UUID languageCode = MetaData.ENGLISH_LANGUAGE.getPrimordialUuid();
			UUID dialect = MetaData.US_ENGLISH_DIALECT.getPrimordialUuid();
			UUID caseSignificance = MetaData.DESCRIPTION_CASE_SENSITIVE.getPrimordialUuid();

			for (CVXInfo row : terminology.getCVXInfo()) {
				try {
					final String code = CVXCodesHelper.getCVXCode(row) + "";
					final String fsn = CVXCodesHelper.getShortDescription(row);
					final String preferred = CVXCodesHelper.getFullVaccinename(row);
					final State state = CVXCodesHelper.getOchreState(row);
					final String cvxStatus = CVXCodesHelper.getStatus(row);
					final long lastUpdateTime = CVXCodesHelper.getLastUpdatedDate(row).getTime();

					// Create row concept
					final UUID rowConceptUuid = ConverterUUID.createNamespaceUUIDFromString(code);
					final ConceptChronology<? extends ConceptVersion<?>> rowConcept = importUtil_.createConcept(rowConceptUuid, lastUpdateTime, state, null);
					final ComponentReference rowComponentReference = ComponentReference.fromConcept(rowConcept);
					importUtil_.addParent(rowComponentReference, cvxRootConcept.getPrimordialUuid());
					
					importUtil_.addDescription(
							rowComponentReference,
							null,
							fsn,
							DescriptionType.FSN,
							true,
							dialect,
							caseSignificance,
							languageCode,
							null,
							fsnSourceDescriptionTypeUUID,
							null,
							lastUpdateTime);
					
					importUtil_.addDescription(
							rowComponentReference,
							null,
							preferred,
							DescriptionType.SYNONYM,
							true,
							dialect,
							caseSignificance,
							languageCode,
							null,
							preferredSynonymSourceDescriptionTypeUUID,
							null,
							lastUpdateTime);

					// Add required CVXCode annotation
					importUtil_.addStaticStringAnnotation(rowComponentReference, code, cvxCodePropertyUuid, null);

					// Add required CVX extended Status annotation
					if (!(cvxStatus.toUpperCase().equals("ACTIVE") || cvxStatus.toUpperCase().equals("INACTIVE")))
					{
						importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeStringImpl(cvxStatus), 
							cvxStatusPropertyUuid, null, lastUpdateTime);
					}
					
					// Add optional Notes comment annotation
					if (StringUtils.isNotBlank(CVXCodesHelper.getNotes(row))) {
						importUtil_.addAnnotation(rowComponentReference, null, new DynamicSememeStringImpl(CVXCodesHelper.getNotes(row)), 
								DynamicSememeConstants.get().DYNAMIC_SEMEME_COMMENT_ATTRIBUTE.getPrimordialUuid(), null, lastUpdateTime);
					}

					// Add to refset allCvxConceptsRefset
					importUtil_.addRefsetMembership(rowComponentReference, allCvxConceptsRefset, null, lastUpdateTime);

					++conceptCount;
				} catch (Exception e) {
					final String msg = "Failed processing row with " + e.getClass().getSimpleName() + " " + e.getLocalizedMessage() + ": " + row;
					ConsoleUtil.printErrorln(msg);
					throw new RuntimeException(msg, e);
				}
			}
			ConsoleUtil.println("Processed " + conceptCount + " concepts");
			
			ConsoleUtil.println("Load Statistics");

			for (String line : importUtil_.getLoadStats().getSummary())
			{
				ConsoleUtil.println(line);
			}
			// this could be removed from final release. Just added to help debug editor problems.
			ConsoleUtil.println("Dumping UUID Debug File");
			ConverterUUID.dump(outputDirectory, "cvxUuid");

			importUtil_.shutdown();
			ConsoleUtil.writeOutputToFile(new File(outputDirectory, "ConsoleOutput.txt").toPath());
		}
		catch (Exception ex)
		{
			throw new MojoExecutionException(ex.getLocalizedMessage(), ex);
		}
	}

	private ConceptChronology<? extends ConceptVersion<?>> createType(UUID parentUuid, String typeName) throws Exception
	{
		ConceptChronology<? extends ConceptVersion<?>> concept = importUtil_.createConcept(typeName, true);
		loadedConcepts.put(concept.getPrimordialUuid(), typeName);
		importUtil_.addParent(ComponentReference.fromConcept(concept), parentUuid);
		return concept;
	}

	public static void main(String[] args) throws MojoExecutionException
	{
		CVXImportMojo i = new CVXImportMojo();
		i.outputDirectory = new File("../cvx-ibdf/target");
		i.inputFileLocation = new File("../cvx-ibdf/target/generated-resources/src/");
		i.converterOutputArtifactVersion = "2016.01.07.foo";
		i.converterVersion = "SNAPSHOT";
		i.converterSourceArtifactVersion = "17.0";
		i.execute();
	}
}